/** \file
 * \brief Windows Driver Controls creation
 *
 * See Copyright Notice in iup.h
 * $Id: wincreat.c,v 1.53 2004/09/18 04:23:48 scuri Exp $
 */

#include <stdio.h>    /* sprintf */
#include <windows.h>
#include <math.h>
#include <assert.h>

#include "iglobal.h"
#include "idrv.h"
#include "win.h"
#include "wintips.h"
#include "wincreat.h"
#include "winproc.h"
#include "winutil.h"
#include "winicon.h"
#include "winbutton.h"
#include "wintoggle.h"
#include "winframe.h"

/* global variables */
HINSTANCE    iupwin_hinstance;

/* prototipos */
static void assignidIhandle(Ihandle *h);
static long gettextalignment(Ihandle *n);

static HWND iupwindialog;
static Ihandle* hdial = NULL;

#define INITIALNUMBER 50
#define MENUINITIALNUMBER 10 /* Start with 10 because then we know that lower numbers
                             ** are not menus (necessary because of DEFAULTENTER and
                             ** DEFAULTESC. */

#define NUM_MAX_MENUS 65000
static  Ihandle* menus[NUM_MAX_MENUS];
static  unsigned int childnumber = INITIALNUMBER;

static long type_of_toggle = BS_AUTOCHECKBOX;  /* toggle options used in radios */
static long togglegroup = 0;
static Ihandle* firsttoggle = NULL;

#define CreateColor(n,x,y)   handle(n) = NULL

#define IUPWIN_SINGLELINE 0
#define IUPWIN_MULTILINE  1

/* variaveis para registro de classes no Windows */
static int dialogcount = 0;
static int canvascount = 0;

void iupwinShowLastError(void)
{
  LPVOID lpMsgBuf;
  FormatMessage( 
    FORMAT_MESSAGE_ALLOCATE_BUFFER | 
    FORMAT_MESSAGE_FROM_SYSTEM | 
    FORMAT_MESSAGE_IGNORE_INSERTS,
    NULL,
    GetLastError(),
    0,
    (LPTSTR) &lpMsgBuf,
    0,
    NULL 
  );
  MessageBox( NULL, (LPCTSTR)lpMsgBuf, "Error", MB_OK | MB_ICONINFORMATION );
  LocalFree( lpMsgBuf );
}

static void sethandle( Ihandle* h, HWND hwnd )
{
  assert(hwnd);
#if _TEST_CODE_ 
  if(hwnd == NULL)
    iupwinShowLastError();
#endif
  handle(h) = hwnd;
}

static int iupEnvCheck(Ihandle *n, char *a)
{
 char* v = iupGetEnv (n, a);
 if (v == NULL)
   v = iupdrvGetDefault(n, a);
 if (v == NULL)
  return NOTDEF;
 else if (strequalnocase(v, IUP_NO) || strequalnocase(v, IUP_OFF))
  return NO;
 else if (strequalnocase(v, IUP_YES) || strequalnocase(v, IUP_ON))
  return YES;
 return NOTDEF;
}

static char* GetDialogClass(Ihandle* h)
{
  char* name = (char*) malloc(sizeof(char)*20);
  WNDCLASS wndclass;
  ZeroMemory(&wndclass, sizeof(WNDCLASS));
  
  /* each window has a different class so we can  */

  sprintf( name, "IUP%i", dialogcount );
  dialogcount++;

  wndclass.hInstance      = iupwin_hinstance;
  wndclass.lpszClassName  = name;
  wndclass.lpfnWndProc    = (WNDPROC)iupwinDialogProc;
  wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
  wndclass.style          = CS_PARENTDC | CS_OWNDC;
  wndclass.hbrBackground  = (HBRUSH)(atoi(IupGetGlobal("WIN_DLGBGCOLOR"))+1);

  if (iupCheck(h,"WIN_SAVEBITS"))
    wndclass.style |= CS_SAVEBITS;

  if (iupCheck(h, IUP_CONTROL) == YES)
    wndclass.style |=  CS_HREDRAW | CS_VREDRAW;
  RegisterClass(&wndclass);

  return name;
}

static char* GetCanvasClass(void)
{
  char* name = (char*) malloc(sizeof(char)*20);
  WNDCLASS wndclass;
  ZeroMemory(&wndclass, sizeof(WNDCLASS));

  sprintf( name, "IUPCnv%i", canvascount );
  canvascount ++;

  wndclass.hInstance      = iupwin_hinstance;
  wndclass.lpszClassName  = name;
  wndclass.lpfnWndProc    = (WNDPROC)iupwinCanvasProc;
  wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
  wndclass.style          = CS_CLASSDC | CS_DBLCLKS; 

  RegisterClass(&wndclass);

  return name;
}

static void InitColors(Ihandle *n, RGBQUAD* colors, int pal_count, int *transp_index)
{
  int i;
  char *value;
  char attr[6];

  static struct{unsigned char r, g, b;} default_colors[16] = {
    { 0,0,0 },     
    { 128,0,0 },
    { 0,128,0 },    
    { 128,128,0 },    
    { 0,0,128 },    
    { 128,0,128 },    
    { 0,128,128 },    
    { 192,192,192 },    
    { 128,128,128 },    
    { 255,0,0 },
    { 0,255,0 },
    { 255,255,0 },
    { 0,0,255 },
    { 255,0,255 },
    { 0,255,255 },
    { 255,255,255 }};

  *transp_index = -1;

  for (i=0;i<16;i++)
  {
    sprintf( attr, "%d", i );
    value = IupGetAttribute(n, attr);

    if(value != NULL)
    {
      int red, green, blue;

      if(strequal(value,"BGCOLOR"))
      {
        /* retrieve the background color */
        value = IupGetAttribute(n, IUP_BGCOLOR);
        if (*transp_index == -1) *transp_index = i;
      }

      if (sscanf(value, "%d %d %d", &red, &green, &blue) != 3)
      {
        colors[i].rgbRed   = default_colors[i].r;
        colors[i].rgbGreen = default_colors[i].g;
        colors[i].rgbBlue  = default_colors[i].b;
      }
      else
      {
        colors[i].rgbRed   = (BYTE)red;
        colors[i].rgbGreen = (BYTE)green;
        colors[i].rgbBlue  = (BYTE)blue;
      }
    }
    else
    {
      colors[i].rgbRed   = default_colors[i].r;
      colors[i].rgbGreen = default_colors[i].g;
      colors[i].rgbBlue  = default_colors[i].b;
    }

    colors[i].rgbReserved = 0;
  }

  for (;i<pal_count;i++)
  {
    int red, green, blue;
    sprintf( attr, "%d", i );
    value = IupGetAttribute(n, attr);

    sscanf(value, "%d %d %d", &red, &green, &blue);
    colors[i].rgbRed   = (BYTE)red;
    colors[i].rgbGreen = (BYTE)green;
    colors[i].rgbBlue  = (BYTE)blue;
  }

  /* If cursor and no transparency defined, assume 0 is transparent. */
  if(IupGetAttribute(n, IUP_HOTSPOT) != NULL && *transp_index == -1)
  {
    *transp_index = 0;
    colors[0].rgbRed   = 0;
    colors[0].rgbGreen = 0;
    colors[0].rgbBlue  = 0;
  }

  IupSetfAttribute(n, "_TRANSP_INDEX", "%d", *transp_index);
}

static void CreateImage (Ihandle *n)
{
  int i,j;
  int width = IupGetInt(n,IUP_WIDTH);
  int height = IupGetInt(n,IUP_HEIGHT);
  int line_size, mask_line;
  unsigned char *data = (unsigned char*)image_data(n);
  int transp_index;

  IwinBitmap* bmp;
  BYTE* aImage;
  BYTE* aMask;
  int pal_count = 16;
  int bpp = 4;

  bmp  = (IwinBitmap*) malloc(sizeof(IwinBitmap));
  handle(n) = bmp;

  for (i=16;i<256;i++)
  {
    char attr[6];
    sprintf( attr, "%d", i );
    if (!IupGetAttribute(n, attr))
      break;
    pal_count++;
  }

  if (pal_count > 16)
    bpp = 8;

  bmp->bmpinfo = malloc(sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD)*pal_count);

  {
    BITMAPINFOHEADER* bmpheader = (BITMAPINFOHEADER*)bmp->bmpinfo;
    memset(bmpheader, 0, sizeof(BITMAPINFOHEADER));
    bmpheader->biSize = sizeof(BITMAPINFOHEADER);
    bmpheader->biWidth  = width;
    bmpheader->biHeight = height;
    bmpheader->biPlanes = 1;
    bmpheader->biBitCount = bpp;
    bmpheader->biCompression = BI_RGB;
    bmpheader->biClrUsed = pal_count;
  }

  InitColors(n, (RGBQUAD*)((BYTE*)bmp->bmpinfo + sizeof(BITMAPINFOHEADER)), pal_count, &transp_index);

  line_size = ((width * bpp + 31) / 32) * 4;   /* DWORD aligned, 4 bytes boundary in a N bpp image */
  mask_line = ((width * 1 + 15) / 16) * 2;     /* WORD aligned, 2 bytes boundary in a 1 bpp image */

  bmp->bitmap = malloc(height * line_size);
  bmp->bitmask = malloc(height * mask_line);

  memset(bmp->bitmap, 0, height * line_size);
  memset(bmp->bitmask, 0, height * mask_line);

  aImage = (BYTE*)bmp->bitmap;   /* windows bitmaps are bottom up */
  aMask = (BYTE*)bmp->bitmask;  

  aMask += (height-1)*mask_line; /* mask is top down */
  data += (height-1)*width;      /* iupimage is top down */

  /* for cursors and icons */
  /* destination = (destination AND bitmask) XOR bitmap */

  for (i=0;i<height;i++)
  {
    for(j=0;j<width;j++)
    {
      unsigned char pixel = data[j];

      /* seta bit na mascara */
      if(transp_index != -1 && transp_index == pixel)
      {
        aMask[j/8] |= 1 << (7 - (j % 8)); /* set mask bit */
      }
      
      if (bpp == 4)
      {
        if (j % 2)
          aImage[j/2] = (unsigned char)((0x0F & pixel) | (0xF0 & aImage[j/2])); /* second 4 bits low */
        else
          aImage[j/2] = (unsigned char)(0xF0 & (pixel << 4)); /* first 4 bits high */
      }
      else
        aImage[j] = pixel;
    }

    aImage += line_size;
    aMask -= mask_line;
    data -= width;
  }
}

static void changeProc(Ihandle *n, WNDPROC new_proc, char* name)
{
  int test;
  HWND hwnd = handle(n);
  IupSetAttribute(n, name, (char*)GetWindowLong (hwnd, GWL_WNDPROC));
  test = SetWindowLong (hwnd, GWL_WNDPROC, (LONG)new_proc);
  assert(test);
}

static void CreateButton (Ihandle *n, int x, int y)
{
 char* image_name      = iupwinCopyStr(IupGetAttribute( n, IUP_IMAGE));
 char* iminactive_name = iupwinCopyStr(IupGetAttribute( n, IUP_IMINACTIVE ));
 char* impress_name    = iupwinCopyStr(IupGetAttribute( n, IUP_IMPRESS ));
 char* bgcolor_name    = iupwinCopyStr(iupGetEnv( n, IUP_BGCOLOR ));

 if (image_name)
 {
   Ihandle* hImage = IupGetHandle( image_name );
   if ( hImage && handle(hImage)==NULL)
     iupdrvCreateObjects( hImage );
 }

 if (iminactive_name)
 {
   Ihandle* hImage = IupGetHandle( iminactive_name );
   if (hImage && handle(hImage)==NULL)
     iupdrvCreateObjects( hImage );
 }

 if (impress_name)
 {
   Ihandle* hImage = IupGetHandle( impress_name );
   if (hImage && handle(hImage)==NULL)
     iupdrvCreateObjects( hImage );
 }

 /* Buttons with the BS_PUSHBUTTON, BS_DEFPUSHBUTTON, or BS_PUSHLIKE styles
    do not use the returned brush in WM_CTLCOLORBTN. 
    Buttons with these styles are always drawn with the default system colors.
    >> So we use ownerdraw buttons because of the BGCOLOR attribute.
 */

 sethandle(n, CreateWindow ("IUPBUTTON",   /* BUTTON class */
        (LPCSTR)IupGetAttribute(n,IUP_TITLE),      /* window name */
        WS_CHILD | WS_TABSTOP | BS_NOTIFY |
        (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0)    |
        BS_OWNERDRAW,        /* window style */
        x,                   /* x-position */
        y,                   /* y-position */
        currentwidth(n),     /* width */
        currentheight(n),    /* height */
        iupwindialog,    /* window parent */
        (HMENU)++childnumber,/* Identifier */
        iupwin_hinstance,           /* instance of app. */
        NULL) );               /* no creation parameters */

 /*redefine a callback de mensagens do botao*/
 changeProc(n, iupwinButtonProc, "__IUPWIN_BUTTONOLDPROC__");

 iupwinSaveIhandle(handle(n), n);

 if (iupEnvCheck(n, IUP_ACTIVE) == NO)
  EnableWindow ((HWND)handle(n), FALSE);

 if(image_name) free(image_name);
 if(iminactive_name) free(iminactive_name);
 if(impress_name) free(impress_name);
 if(bgcolor_name) free(bgcolor_name);
}

void iupwinSetHScrollInfo( Ihandle* h )
{
  float posx=0.0F, xmin=0.0F, xmax=1.0F;
  float ratio;
  unsigned short x;
  float dx;
  SCROLLINFO scrollinfo;

   if (IupGetAttribute(h,IUP_XMIN))
      xmin = IupGetFloat(h,IUP_XMIN);

   if (IupGetAttribute(h,IUP_XMAX))
      xmax = IupGetFloat(h,IUP_XMAX);

   if (IupGetAttribute(h,IUP_POSX))
      posx = IupGetFloat(h,IUP_POSX);

   ratio = (posx-xmin)/(xmax-xmin);
   x = (unsigned short) (HORZ_SCROLLBAR_SIZE * ratio);
  if ( x < 1 )
    x = 1;

  dx = IupGetFloat(h,IUP_DX);
  if (dx == 0.0F)
    dx = 1.0F;
  scrollinfo.cbSize = sizeof(SCROLLINFO);
  scrollinfo.fMask = SIF_ALL;
  scrollinfo.nPage = (int) ((dx/(xmax-xmin))*HORZ_SCROLLBAR_SIZE);

  if (scrollinfo.nPage < 1)
    scrollinfo.nPage = 1;
  else if (scrollinfo.nPage > HORZ_SCROLLBAR_SIZE)
    {
       x = 1;
    scrollinfo.nPage = HORZ_SCROLLBAR_SIZE;
    }

    scrollinfo.nPos = x;
    scrollinfo.nMax = HORZ_SCROLLBAR_SIZE;
    scrollinfo.nMin = 1;

  SetScrollInfo((HWND)handle(h), SB_HORZ, &scrollinfo, TRUE);
}

void iupwinSetVScrollInfo( Ihandle* h )
{
  float posy=0.0F, ymin=0.0F, ymax=1.0F;
  float ratio;
  unsigned short y;

  if (IupGetAttribute(h,IUP_YMIN))
    ymin = IupGetFloat(h,IUP_YMIN);

  if (IupGetAttribute(h,IUP_YMAX))
    ymax = IupGetFloat(h,IUP_YMAX);

  if (IupGetAttribute(h,IUP_POSY))
    posy = IupGetFloat(h,IUP_POSY);

  ratio = (posy-ymin)/(ymax-ymin);
  y = (unsigned short) (VERT_SCROLLBAR_SIZE * ratio);
  if ( y < 1 )
    y = 1;

  {
    float dy = IupGetFloat(h,IUP_DY);
    SCROLLINFO scrollinfo;
    if (dy == 0.0F)
      dy = 1.0F;
    scrollinfo.cbSize = sizeof(SCROLLINFO);
    scrollinfo.fMask = SIF_ALL;
    scrollinfo.nPage = (int) ((dy/(ymax-ymin))*VERT_SCROLLBAR_SIZE);

    if (scrollinfo.nPage < 1)
      scrollinfo.nPage = 1;
    else if (scrollinfo.nPage > VERT_SCROLLBAR_SIZE)
      {
         scrollinfo.nPage = VERT_SCROLLBAR_SIZE;
         y = 1;
      }

      scrollinfo.nPos = y;
      scrollinfo.nMax = VERT_SCROLLBAR_SIZE;
      scrollinfo.nMin = 1;

    SetScrollInfo((HWND)handle(h), SB_VERT, &scrollinfo, TRUE);
  }
}

static void enableDragDrop(Ihandle *n)
{
  char *teste;
  teste = IupGetAttribute(n,IUP_DROPFILES_CB);
  if (teste!=NULL)
    DragAcceptFiles((HWND)handle(n), TRUE );
  else 
    DragAcceptFiles((HWND)handle(n), FALSE );
}

static void CreateCanvas (Ihandle *n, int x, int y)
{
 char buffer[30];
 char* classname = NULL;
 
 DWORD sbstyle = 0L;

 if (IupGetAttribute(n,IUP_SCROLLBAR) && strequal(IupGetAttribute(n,IUP_SCROLLBAR),IUP_YES))
   sbstyle = WS_VSCROLL | WS_HSCROLL;
 else
 {
   if ( (IupGetAttribute(n,IUP_SBH) && strequal(IupGetAttribute(n,IUP_SBH),IUP_YES)) ||
        (IupGetAttribute(n,IUP_SCROLLBAR) && strequal(IupGetAttribute(n,IUP_SCROLLBAR),IUP_HORIZONTAL)) )
     sbstyle = WS_HSCROLL;
   if ( (IupGetAttribute(n,IUP_SBV) && strequal(IupGetAttribute(n,IUP_SBV),IUP_YES)) ||
        (IupGetAttribute(n,IUP_SCROLLBAR) && strequal(IupGetAttribute(n,IUP_SCROLLBAR),IUP_VERTICAL)) )
     sbstyle |= WS_VSCROLL;
 }

 if (handle(n))
 {
   GetClassName( (HWND)handle(n), buffer, 30 );
   classname = strdup(buffer);
 }
 else
   classname = GetCanvasClass();

 sethandle(n,CreateWindow(classname,   /* Canvas class */
        "canvas",            /* window name */
        (iupCheck(n, IUP_BORDER) ? WS_BORDER : 0) |
        (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
        sbstyle | WS_TABSTOP | WS_CHILD,    /* window style */
        x,                   /* x-position */
        y,                   /* y-position */
        currentwidth(n),     /* width */
        currentheight(n),    /* height */
        iupwindialog,    /* window parent */
        (HMENU)++childnumber,           /* Identifier */
        iupwin_hinstance,           /* instance of app. */
        NULL));               /* no creation parameters */

 if(classname)
    free(classname);

 iupwinSaveIhandle(handle(n), n);

 iupwinSetBgColor (n);
 iupwinSetCursor (n);

 if (sbstyle & WS_HSCROLL)
   iupwinSetHScrollInfo(n);
 else
   IupSetAttribute(n,IUP_POSX,"0.0");

 if (sbstyle & WS_VSCROLL)
   iupwinSetVScrollInfo(n);
 else
   IupSetAttribute(n,IUP_POSY,"0.0");

 if (iupEnvCheck(n, IUP_ACTIVE) == NO)
   EnableWindow ((HWND)handle(n), FALSE);

 enableDragDrop(n);
}

static void CreateDlg (Ihandle *n, int x, int y)
{
 DWORD error;
 HMENU menu = iupwinGetMenu (n);
 DWORD dialstyle = 0, dialexstyle = 0;
 char *classname;
 char *clip = NULL;
 HWND hwnd_owner=NULL;
 int titlebar  = 0,
     hasborder = 0;

 if(iupGetEnv(n, IUP_TITLE))
   titlebar = 1;

 if (iupCheck(n,IUP_MAXBOX))
 {
   dialstyle |= WS_MAXIMIZEBOX;
   titlebar = 1;
 }

 if (iupCheck(n,IUP_MINBOX))
 {
   dialstyle |= WS_MINIMIZEBOX;
   titlebar = 1;
 }

 if (iupCheck(n,IUP_MENUBOX))
 {
   dialstyle |= WS_SYSMENU;
   titlebar = 1;
 }

 if (iupCheck(n,IUP_RESIZE))
   dialstyle |= WS_THICKFRAME;

 if (!iupCheck(n,IUP_ACTIVE))
   dialstyle |= WS_DISABLED;

 if(iupCheck(n, IUP_BORDER) || titlebar==1)
   hasborder = 1;

 {
   char *name = NULL;
   Ihandle *parent = NULL;

   name  = IupGetAttribute(n, IUP_PARENTDIALOG);

   if(name != NULL)
   {
     parent = IupGetHandle(name);

     if(parent != NULL && type(parent) == DIALOG_)
     {
       if(handle(parent) != NULL)
         hwnd_owner = handle(parent);
       else
       {
         IupMap(parent);
         hwnd_owner = handle(parent);
       }
     }
     
     dialstyle |= WS_POPUP;

     if(titlebar == 1)
       dialstyle |= WS_CAPTION;
     else if (hasborder == 1)
       dialstyle |= WS_BORDER;
   }
   else
   {
     if (titlebar == 1)
       dialstyle |= WS_OVERLAPPED;
     else if(titlebar == 0 && hasborder == 1)
       dialstyle |= WS_POPUP | WS_BORDER;
     else
       dialstyle |= WS_POPUP;
   }
 }

 if(hwnd_owner == NULL) /* if user hasn't set IUP_PARENTDIALOG */
  hwnd_owner = (HWND) IupGetAttribute(n, "NATIVEPARENT");

 if(iupCheck(n, IUP_TOOLBOX)==YES && hwnd_owner!=NULL)
   dialexstyle |= WS_EX_TOOLWINDOW | WS_EX_WINDOWEDGE;

 if(iupCheck(n, IUP_CONTROL)==YES && hwnd_owner!=NULL) {
   dialexstyle |= WS_EX_CONTROLPARENT;
   dialstyle = WS_CHILD | WS_TABSTOP | WS_CLIPCHILDREN;
 }

 if(iupCheck(n, "HELPBUTTON")==YES)
   dialexstyle |= WS_EX_CONTEXTHELP;

 classname = GetDialogClass(n);

 clip = IupGetAttribute(n, IUP_CLIPCHILDREN);
 if(clip && strequal(clip, IUP_YES))
   dialstyle = dialstyle | WS_CLIPCHILDREN; /* Clips children. Must warn them to repaint */
 
 handle(n) = (void *) CreateWindowEx (dialexstyle /* extended styles */, 
        classname,  /* class */
		titlebar ? iupGetEnv(n, IUP_TITLE) : NULL, /* title */
        dialstyle |              /* style */
        WS_CLIPSIBLINGS,
        CW_USEDEFAULT,           /* x-position */
        CW_USEDEFAULT,           /* y-position */
        currentwidth(n)+iupwinDialogDecorX(n),
        currentheight(n)+iupwinDialogDecorY(n),
        hwnd_owner,                /* owner window */
        menu,              /* Menu */
        iupwin_hinstance,           /* instance of app. */
        NULL);               /* no creation parameters */

 error = GetLastError();

 free(classname);
 
 iupwinSaveIhandle(handle(n), n);

 enableDragDrop(n);

 iupwinSetBgColor (n);
 iupwinSetCursor (n);
 iupwinUpdateAttr(n, "TRAY");
 iupwinUpdateAttr(n, "TRAYTIP");
 iupwinUpdateAttr(n, "TRAYIMAGE");
}

static void CreateBox (Ihandle *n, int x, int y)
{
 assert(iupwindialog);
 handle(n) = (void *) iupwindialog;
}


static void CreateFill (Ihandle *n, int x, int y)
{
 assert(iupwindialog);
 handle(n) = (void *) iupwindialog;
}

static void CreateFrame (Ihandle *n, int x, int y)
{
 char *clip = IupGetAttribute(n, IUP_CLIPCHILDREN);

 if (iupGetEnv(n, IUP_TITLE))
  sethandle(n, CreateWindow ("IUPBUTTON",   /* BUTTON class */
        (LPCSTR)iupGetEnv(n, IUP_TITLE),  /* window name */
        WS_CHILD | 
        (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
        BS_GROUPBOX,             /* window style */
        x,                       /* x-position */
        y,                       /* y-position */
        currentwidth(n),         /* width */
        currentheight(n),        /* height */
        iupwindialog,        /* window parent */
        (HMENU)++childnumber,        /* Identifier */
        iupwin_hinstance,               /* instance of app. */
        NULL));                   /* no creation parameters */
 else
  sethandle(n, CreateWindow ("IUPSTATIC",   /* STATIC class */
        NULL,  /* window name */
        WS_CHILD |
        (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
        SS_ETCHEDFRAME,           /* window style */
        x,                       /* x-position */
        y,                       /* y-position */
        currentwidth(n),         /* width */
        currentheight(n),        /* height */
        iupwindialog,        /* window parent */
        (HMENU)++childnumber,        /* Identifier */
        iupwin_hinstance,               /* instance of app. */
        NULL));                   /* no creation parameters */

 if (iupEnvCheck(n, IUP_ACTIVE) == NO)
  EnableWindow ((HWND)handle(n), FALSE);

 iupwinSaveIhandle(handle(n), n);

 if(clip && strequal(clip, IUP_YES))
 {
   /* Setting frame's own window proc. Necessary because it does not paint its
    * own background. Should always be done after setting the Ihandle, so that the
    * value can be retrieved. */
   changeProc(n, iupwinFrameProc, "__IUPWIN_FRAMEOLDPROC__");
 }
}

static void CreateLabel (Ihandle *n, int x, int y)
{
 Ihandle* hImage;
 int separator = 0;
 char* image_name = IupGetAttribute(n, IUP_IMAGE);
 long alignment = gettextalignment(n);

/* the owner draw implementation is only necessary because of the background color */

 if (image_name)
 {
   hImage = IupGetHandle( image_name );
   if (hImage && handle(hImage)==NULL)
     iupdrvCreateObjects( hImage );
 }

 {
    char* sep = IupGetAttribute(n, IUP_SEPARATOR);
    if (sep)
    {
      if (strequal(sep, IUP_HORIZONTAL))
        separator = 1;
      else if (strequal(sep, IUP_VERTICAL))
        separator = 2;
    }
 }

 if (separator)
  sethandle(n, CreateWindow ( "IUPSTATIC",   /* STATIC class */
          NULL,   /* window name */
          (separator == 1? SS_ETCHEDHORZ: SS_ETCHEDVERT) |
          (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
          WS_CHILD,             /* window style */
          x,                   /* x-position */
          y,                   /* y-position */
          currentwidth(n),     /* width */
          currentheight(n),    /* height */
          iupwindialog,    /* window parent */
          (HMENU)++childnumber,    /* Identifier */
          iupwin_hinstance,           /* instance of app. */
          NULL));               /* no creation parameters */
 else
  sethandle(n, CreateWindow ( (image_name?"IUPBUTTON":"IUPSTATIC"),   /* STATIC class */
          (LPCSTR)IupGetAttribute(n,IUP_TITLE),   /* window name */
          (image_name?BS_OWNERDRAW:alignment) |   /* set alignment only for non owner draw */
          (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
          WS_CHILD,             /* window style */
          x,                   /* x-position */
          y,                   /* y-position */
          currentwidth(n),     /* width */
          currentheight(n),    /* height */
          iupwindialog,    /* window parent */
          (HMENU)++childnumber,    /* Identifier */
          iupwin_hinstance,           /* instance of app. */
          NULL));               /* no creation parameters */

 if (image_name)
   EnableWindow ((HWND)handle(n), 0);

 iupwinSaveIhandle(handle(n), n);
 iupwinUpdateFont(n);

 if (iupEnvCheck(n, IUP_ACTIVE) == NO)
  EnableWindow ((HWND)handle(n), FALSE);
}

static void CreateList (Ihandle *n, int x, int y)
{
  int iscombobox;
  int isdropdown = iupCheck(n, IUP_DROPDOWN)==YES? 1: 0;
  int haseditbox = iupCheck(n, "EDITBOX")==YES? 1: 0;

  if (isdropdown || haseditbox)
  {
    sethandle(n, CreateWindowEx (WS_EX_CLIENTEDGE, "IUPCOMBOBOX",   /* COMBOBOX class */
      "iupcombo",            /* window name */
      (iupEnvCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) | 
      (haseditbox? (isdropdown? CBS_DROPDOWN: CBS_SIMPLE)|CBS_AUTOHSCROLL: CBS_DROPDOWNLIST) | 
      WS_TABSTOP | WS_CHILD | WS_VSCROLL,    /* window style */
      x,                   /* x-position */
      y,                   /* y-position */
      CW_USEDEFAULT,       /* width */
      CW_USEDEFAULT,       /* height */
      iupwindialog,       /* window parent */
      (HMENU)++childnumber,    /* Identifier */
      iupwin_hinstance,           /* instance of app. */
      NULL));               /* no creation parameters */

  }
  else
  {
    sethandle(n, CreateWindowEx (WS_EX_CLIENTEDGE,"IUPLISTBOX",   /* LISTBOX class */
      "iuplistbox",            /* window name */
      (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
      (iupCheck(n, IUP_MULTIPLE)==YES ? LBS_EXTENDEDSEL : 0) |
      LBS_NOTIFY |
      WS_TABSTOP | WS_CHILD | WS_VSCROLL,  /* window style */
      x,                   /* x-position */
      y,                   /* y-position */
      currentwidth(n),     /* width */
      currentheight(n),    /* height */
      iupwindialog,    /* window parent */
      (HMENU)++childnumber,           /* Identifier */
      iupwin_hinstance,           /* instance of app. */
      NULL));               /* no creation parameters */
  }

  iupwinSaveIhandle(handle(n), n);

  iscombobox = isdropdown|haseditbox;

  {  /* insere strings na lista */
    char  op[5];
    char *v;
    int i=1;
    UINT addstring = iscombobox? CB_ADDSTRING : LB_ADDSTRING;

    while(1)
    {
      sprintf (op, "%d", i);
      v = iupGetEnv (n,op);
      if (v == NULL)
        break;
      SendMessage ((HWND)handle(n), addstring, 0, (DWORD)v);
      i++;
    }
  }

  {  /* inicializa opcoes da lista */
    char* v = iupGetEnv (n, IUP_VALUE);
    UINT setcursel = iscombobox? CB_SETCURSEL : LB_SETCURSEL;
    unsigned  i;
    if (v)
    {
      if (iupCheck(n, IUP_MULTIPLE)==YES)
      {
        for (i=0; i < strlen(v); ++i)
          if (v[i] == '+')
            SendMessage ((HWND)handle(n), LB_SETSEL, 1, i);
      }
      else
      {
        sscanf (v, "%d", &i);
        SendMessage ((HWND)handle(n), setcursel, i-1, 0L);
      }
    }
    else if (iscombobox)
      SendMessage ((HWND)handle(n), setcursel, 0, 0L);
  }

  if (iupEnvCheck(n, IUP_ACTIVE) == NO)
    EnableWindow ((HWND)handle(n), FALSE);

  if (haseditbox)
  {
    char* readonly;
    COMBOBOXINFO boxinfo;
    char* strMaxChar = iupGetEnv( n, IUP_NC );
    if(strMaxChar)
    {
      int maxchar = atoi(strMaxChar);
      SendMessage( (HWND)handle(n), CB_LIMITTEXT, maxchar, 0L );
    }

    ZeroMemory(&boxinfo, sizeof(COMBOBOXINFO));
    boxinfo.cbSize = sizeof(COMBOBOXINFO);
    GetComboBoxInfo((HWND)handle(n), &boxinfo);
    iupwinSaveIhandle(boxinfo.hwndItem, n);
    IupSetAttribute(n, "_EDITBOX_IUPWIN", (char*)boxinfo.hwndItem);

    IupSetAttribute(n, "__IUPWIN_TEXTOLDPROC__", (char*)GetWindowLong(boxinfo.hwndItem, GWL_WNDPROC));
    SetWindowLong (boxinfo.hwndItem, GWL_WNDPROC, (LONG)iupwinEditProc);

    readonly = iupGetEnv( n, IUP_READONLY );
    if(readonly && strequal(readonly, "YES"))
      SendMessage(boxinfo.hwndItem, EM_SETREADONLY, (WPARAM)(BOOL) TRUE, 0);
  }
}

char* iupwinMenuLabel( Ihandle* h, char* label )
{
  char* keyattr = IupGetAttribute( h, IUP_KEY );

  if (keyattr)
  {
    int keycode;
    int size;
    int i, j;
    char *newlabel = (char*)malloc(sizeof(char)*strlen(label)+3);

    keycode = iupKeyDecode(keyattr);
    size = strlen(label) + 1;
    for (i=0, j=0; i < size; i++, j++)
    {
      if (label[i]==keycode)
      {
        newlabel[j] = '&';
        j++;
        strcpy(newlabel + j, label + i);
        break;
      }
      newlabel[j] = label[i];
    }

    IupStoreAttribute(h, "WIN_MENULABEL", newlabel);
    free(newlabel);
    return IupGetAttribute(h, "WIN_MENULABEL");
  }
  else
  {
    IupStoreAttribute(h, "WIN_MENULABEL", label);
    return IupGetAttribute(h, "WIN_MENULABEL");
  }
}

static HMENU iupwinFillMenu (Ihandle* n)
{
 HMENU m = (HMENU) handle(n);
 Ihandle *c;

 assert(m);
 assignidIhandle(n);

 foreachchild (c,n)
 {
  assignidIhandle(c);
  if(type(c) == ITEM_)
  {
    Ihandle* himage;
    char *value;
    handle(c) = (void *) m;
    assert(number(c)>0);

    AppendMenu (m, MF_STRING, number(c), iupwinMenuLabel(c, IupGetAttribute(c, IUP_TITLE)));

    himage = IupGetHandle(IupGetAttribute(c, IUP_IMAGE));
    if (himage)
    {
      MENUITEMINFO iteminfo;
      HBITMAP hbm, hbmpress = NULL;
      HDC hDC;             
      Ihandle* himpress;

      himpress = IupGetHandle(IupGetAttribute(c, IUP_IMPRESS));

      hDC = GetDC(NULL);
      hbm = iupwinCreateBitmap(himage, hDC, c);
      if (himpress)
        hbmpress = iupwinCreateBitmap(himpress, hDC, c);
      ReleaseDC(NULL, hDC);

      iteminfo.cbSize = sizeof(MENUITEMINFO);
      iteminfo.fMask = MIIM_CHECKMARKS;
      iteminfo.hbmpUnchecked = hbm;     /* will be deleted by the system */
      iteminfo.hbmpChecked = hbmpress;  

      SetMenuItemInfo(m, number(c), FALSE, &iteminfo);
    }

    if (iupEnvCheck(c,IUP_ACTIVE) == NO)
      EnableMenuItem (m, number(c), MF_GRAYED);

    value = iupGetEnv(c,IUP_VALUE);
    assert(number(c)>0);
    if (value && strequal(value,IUP_ON))
      CheckMenuItem ((HMENU)handle(c), number(c), MF_CHECKED | MF_BYCOMMAND);

  }
  else if(type(c) == MENU_ || type(c) == SUBMENU_)
  {
    int value = 0;
    BOOL test;
    handle(c) = iupwinCreatePopupMenu(child(c));

    if (iupEnvCheck(c,IUP_ACTIVE) == NO)
      value = MF_GRAYED;
     
    test = AppendMenu (m, MF_POPUP | value, 
      (UINT) handle(c), 
      (LPCSTR)iupwinMenuLabel(c, IupGetAttribute(c,IUP_TITLE)));

    assert(test != 0);
  }
  else if(type(c) == SEPARATOR_)
    AppendMenu (m, MF_SEPARATOR, number(c), NULL);
 }
 return m;
}

static HMENU iupwinCreateMenu (Ihandle* n);

HMENU iupwinCreatePopupMenu (Ihandle* n)
{
 handle(n) = (void*)CreatePopupMenu ();
 assert(handle(n));

 iupwinFillMenu( n );

 return (HMENU) handle(n);
}

static HMENU iupwinCreateMenu (Ihandle* n)
{
  handle(n) = (void*)CreateMenu ();
  assert(handle(n));

  iupwinFillMenu( n );

  return (HMENU) handle(n);
}

static void CreateTypein (Ihandle* n, int x, int y, int type)
{
 int to_free;
 char *v = iupwinTransMultiline(iupGetEnv (n, "VALUE"), &to_free);
 char* strMaxChar, *tabsize = NULL;

 if (!v)
  v = "";

 sethandle(n, CreateWindowEx (WS_EX_CLIENTEDGE,"IUPEDIT",     /* EDIT class */
        (LPCSTR)v,                   /* window name */
        ((type==IUPWIN_MULTILINE) ? (ES_MULTILINE | ES_WANTRETURN | WS_HSCROLL | WS_VSCROLL) : 0) |
        ES_LEFT | ES_AUTOVSCROLL | ES_AUTOHSCROLL |
        (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
        (iupCheck(n,IUP_READONLY)==YES ? ES_READONLY : 0) |
        WS_TABSTOP | WS_CHILD,  /* window style */
        x,                   /* x-position */
        y,                   /* y-position */
        currentwidth(n),     /* width */
        currentheight(n),    /* height */
        iupwindialog,    /* window parent */
        (HMENU)++childnumber,    /* Identifier */
        iupwin_hinstance,           /* instance of app. */
        NULL));               /* no creation parameters */

  iupwinSaveIhandle(handle(n), n);

  /*redefine a callback de mensagens do controle de edicao*/
  changeProc(n, iupwinEditProc, "__IUPWIN_TEXTOLDPROC__");

 strMaxChar = iupGetEnv( n, IUP_NC );
 if(strMaxChar)
 {
   int maxchar = atoi(strMaxChar);
   SendMessage( (HWND)handle(n), EM_LIMITTEXT, maxchar, 0L );
 }
 if (iupEnvCheck(n, IUP_ACTIVE) == NO)
  EnableWindow ((HWND)handle(n), FALSE);

 if (type(n) == MULTILINE_)
 {
    tabsize = iupGetEnv(n, "TABSIZE");
    if (tabsize)
       iupdrvSetAttribute(n, "TABSIZE", tabsize);
 }

 if (to_free)
   free(v);
}

static void CreateToggle (Ihandle *n, int x, int y)
{     
 long  last_toggle     = type_of_toggle;
 char* image_name      = IupGetAttribute( n, IUP_IMAGE);
 char* iminactive_name = IupGetAttribute( n, IUP_IMINACTIVE );
 char* impress_name    = IupGetAttribute( n, IUP_IMPRESS );
 long  btstyle = 0;

/* The BGCOLOR for toggles has the same problem of buttons and labels,
   but here we opt to not implement an OWNERDRAW and keep the problem. */

 if (image_name)
 {
   Ihandle* hImage = IupGetHandle( image_name );
   type_of_toggle = type_of_toggle | BS_BITMAP | BS_PUSHLIKE;
   if ( hImage && handle(hImage)==NULL)
     iupdrvCreateObjects( hImage );
 }

 if (iminactive_name)
 {
   Ihandle* hImage = IupGetHandle( iminactive_name );
   if (hImage && handle(hImage)==NULL)
     iupdrvCreateObjects( hImage );
 }

 if (impress_name)
 {
   Ihandle* hImage = IupGetHandle( impress_name );
   if (hImage && handle(hImage)==NULL)
     iupdrvCreateObjects( hImage );
   /* type_of_toggle = type_of_toggle | BS_FLAT; FIXME: removed because non owner draw still contains the border. */
 }

 btstyle = WS_CHILD | BS_NOTIFY |
           (iupCheck(n,IUP_VISIBLE) ? WS_VISIBLE : 0) |
           togglegroup | type_of_toggle;

 /* Do not set TABSTOP for toggles in a radio, 
    we will set it later. */
 if (togglegroup == 0 && firsttoggle == NULL)
   btstyle |= WS_TABSTOP;

 sethandle(n, CreateWindow ("IUPBUTTON",   /* BUTTON class */
        (LPCSTR)IupGetAttribute(n,IUP_TITLE),      /* window name */
        btstyle,             /* windows style */
        x,                   /* x-position */
        y,                   /* y-position */
        currentwidth(n),     /* width */
        currentheight(n),    /* height */
        iupwindialog,    /* window parent */
        (HMENU)++childnumber,    /* Identifier */
        iupwin_hinstance,           /* instance of app. */
        NULL));               /* no creation parameters */

 iupwinSaveIhandle(handle(n), n);

 /* We only need to change procedure if image is wanted */
 if(image_name != NULL)
   changeProc(n, iupwinToggleProc, "__IUPWIN_TOGGLEOLDPROC__");

  /* if we are the first toggle of a radio */
  if (togglegroup != 0)
  {
    /* We use the group style only to the first control in the group */
    togglegroup = 0;
    /* Save the first toggle to set it as the selected one in case the user has
       not chosen one */
    firsttoggle = n;
  }

 iupwinUpdateAttr(n, IUP_VALUE);
 iupwinUpdateAttr(n, IUP_IMAGE);
 iupwinUpdateAttr(n, IUP_ACTIVE);
 type_of_toggle = last_toggle;
}

HMENU iupwinGetMenu (Ihandle* n)
{
 char* menu = iupGetEnv (n, IUP_MENU);
 Ihandle* c;

 if (menu == NULL)
  return NULL;

 c = IupGetHandle (menu);
 if (c == NULL)
  return NULL;

 IupSetAttribute( c, IUPWIN_DIALOG, (char*)n );
 parent(c) = n;
 iupdrvCreateObjects (c);
 return (HMENU) handle(c);
}

void iupdrvCreateObject( Ihandle* self, Ihandle* parent )
{
   HWND olddlg = iupwindialog;
   Ihandle* oldhandle = hdial;

   hdial = parent;
   if (parent)
       iupwindialog = (HWND)handle(parent);
   iupdrvCreateObjects(self);
   iupwindialog = olddlg;
   hdial = oldhandle;
}

void iupdrvCreateObjects (Ihandle *n)
{
  int flag =0;

  if (type(n) != DIALOG_ && iupwindialog == NULL && type(n) != COLOR_ && type(n) != IMAGE_ &&
      type(n) != MENU_)
  {
    flag = 1;
    hdial = IupGetDialog(n);
    iupwindialog = (HWND)handle(hdial);
  }

  if (hclass(n))
    iupCpiMap(n, hdial);
  else
    iupdrvCreateNativeObject(n);
  
  if (flag)
  {
    flag = 0;
    iupwindialog = NULL;
    hdial  = NULL;
  }
}

void iupdrvCreateNativeObject (Ihandle *n)
{
  Ihandle *c;

  if(type(n) == COLOR_)
  {
    if (!handle(n)) CreateColor  (n, posx(n), posy(n));
  }
  else if(type(n) == IMAGE_)
    CreateImage(n);
  else if(type(n) == BUTTON_)
    CreateButton (n, posx(n), posy(n));
  else if(type(n) == CANVAS_)
    CreateCanvas (n, posx(n), posy(n));
  else if(type(n) == DIALOG_)
  {
    CreateDlg (n, posx(n), posy(n));
    iupwindialog = (HWND) handle(n);
    hdial  = n;
    iupdrvCreateObjects (child(n));
    iupwindialog = NULL;
    hdial  = NULL;
  }
  else if(type(n) == FILL_)
    CreateFill (n, posx(n), posy(n));
  else if(type(n) == FRAME_)
  {
    CreateFrame (n, posx(n), posy(n));
    iupdrvCreateObjects (child(n));
  }
  else if(type(n) == HBOX_)
  {
    CreateBox (n, posx(n), posy(n));
    foreachchild(c,n)
      iupdrvCreateObjects (c);
  }
  else if(type(n) == LABEL_)
    CreateLabel (n, posx(n), posy(n));
  else if(type(n) == LIST_)
    CreateList (n, posx(n), posy(n));
  else if(type(n) == MENU_)
    iupwinCreateMenu (n);
  else if(type(n) == RADIO_)
  {
    Ihandle* last_toggle;
    handle(n) = (void *) iupwindialog;
    
    firsttoggle = NULL;
    togglegroup = WS_GROUP; /* this is specified only for the first toggle in the radio. But necessary. */
                            /* this affects keyboard navigation in the dialog for the arrow keys */
                            /* it will form a group up to the next Radio. This is weird. */

    type_of_toggle = BS_AUTORADIOBUTTON;
    iupdrvCreateObjects (child(n));
    type_of_toggle = BS_AUTOCHECKBOX;

    /* Sets checked toggle in this attributed (needed for unchecking purposes) */
    last_toggle = (Ihandle*)iupGetEnv(n,"WIN_LASTTOGGLE");
    if(last_toggle == NULL)
    {
      iupSetEnv(n,"WIN_LASTTOGGLE", (char*) firsttoggle);
      last_toggle = firsttoggle;
    }

    /* sets the tabstop only for the checked toggle */
    SetWindowLong((HWND)handle(last_toggle), GWL_STYLE, GetWindowLong((HWND)handle(last_toggle), GWL_STYLE)|WS_TABSTOP);

    firsttoggle = NULL;
  }
  else if(type(n) == MULTILINE_)
    CreateTypein (n, posx(n), posy(n), IUPWIN_MULTILINE);
  else if(type(n) == TEXT_)
    CreateTypein (n, posx(n), posy(n), IUPWIN_SINGLELINE);
  else if(type(n) == TOGGLE_)
    CreateToggle (n, posx(n), posy(n));
  else if(type(n) == ZBOX_)
  {
    Ihandle *v = NULL;
    char *value = NULL;
    
    value = IupGetAttribute (n,IUP_VALUE);

    /* se value nao tiver sido setado, usa nome do
      primeiro elemento como value default */

    if(value == NULL) 
      value = IupGetName(child(n));

    assert(value != NULL); /* todos os filhos da ZBOX devem ter nomes */
    if(value == NULL)
      return;

    v = IupGetHandle (value);

    CreateBox (n, posx(n), posy(n));
    
    iupStoreEnv(n, "DRV_ZBOX_OLDVALUE", value);
    foreachchild(c,n)
    {
      if (c!=v) 
      {
        iupSetEnv(c,IUP_VISIBLE,IUP_NO);
        iupdrvCreateObjects (c);
      }
      else
        iupdrvCreateObjects (c);
    }
  }
  else if(type(n) == VBOX_)
  {
    CreateBox (n, posx(n), posy(n));
    foreachchild(c,n)
      iupdrvCreateObjects (c);
  }

  iupwinUpdateFont(n);
  iupwinUpdateWinfont(n);
  iupwinUpdateIcon(n);
  iupwinTipsUpdate(n);
  iupwinUpdateTopmost(n);
  iupwinUpdateAttr(n, IUP_FULLSCREEN);
}

void iupwinUpdateAttr(Ihandle *n, char *attr)
{
  char *v;
  assert(n);
  if(n==NULL)
    return;

  assert(attr);
  if(attr == NULL)
    return;

  v = iupGetEnv(n, attr);
  if(v)
  {
    char *newv = strdup(v);
    /* This is necessary because Iup uses the same buffer for
     * iupGetEnv and IupStoreAttibute */
    iupwinSet(n, attr, newv);
    free(newv);
  }

}

/******************************************************************** 
** Assignes a child window number to an Ihandle. Necessary for
** menu items. Assigns the first array position not being used
** by the application.
********************************************************************/
static void assignidIhandle(Ihandle *h)
{
  int i=MENUINITIALNUMBER;
  while(1)
  {
    if(!menus[i])
    {
      menus[i] = h;
      number(h) = i;
      break;
    }
    else
    {
      i++;
      assert(i<NUM_MAX_MENUS); /* Array size is over */
    }
  }
}

/******************************************************************** 
** Removes an id Ihandle association (used by delete item).
********************************************************************/
void iupwinCleanidIhandle(int id)
{
  assert(id>=MENUINITIALNUMBER&&id<NUM_MAX_MENUS);
  if (id<MENUINITIALNUMBER||id>=NUM_MAX_MENUS)
    return;
  assert(menus[id]); /* Must clean an already defined Ihandle */
  menus[id] = NULL;
}

/******************************************************************** 
** Returns a menu Ihandle from a menu window id.
********************************************************************/
Ihandle *iupwinGetMenuIhandle(int id)
{
  assert(id>=MENUINITIALNUMBER&&id<NUM_MAX_MENUS);
  if (id<MENUINITIALNUMBER||id>=NUM_MAX_MENUS)
    return NULL;
  assert(menus[id]);
  return menus[id];
}

/******************************************************************** 
** Initializes this module (called from IupOpen).
********************************************************************/
void iupwinCreatInit(void)
{
  WNDCLASSEX newclass;
  ATOM test = 0;

  memset(menus, 0, NUM_MAX_MENUS*sizeof(Ihandle*));

  memset(&newclass, 0, sizeof(WNDCLASSEX));
  newclass.cbSize         = sizeof(WNDCLASSEX);
  test = GetClassInfoEx(iupwin_hinstance, "EDIT", &newclass);
  assert(test != 0);
  newclass.hInstance      = iupwin_hinstance;
  newclass.lpszClassName  = "IUPEDIT";
  test = RegisterClassEx(&newclass);
  assert(test != 0);

  memset(&newclass, 0, sizeof(WNDCLASSEX));
  newclass.cbSize         = sizeof(WNDCLASSEX);
  test = GetClassInfoEx(iupwin_hinstance, "BUTTON", &newclass);
  assert(test != 0);
  newclass.hInstance      = iupwin_hinstance;
  newclass.lpszClassName  = "IUPBUTTON";
  test = RegisterClassEx(&newclass);
  assert(test != 0);

  memset(&newclass, 0, sizeof(WNDCLASSEX));
  newclass.cbSize         = sizeof(WNDCLASSEX);
  test = GetClassInfoEx(iupwin_hinstance, "STATIC", &newclass);
  assert(test != 0);
  newclass.hInstance      = iupwin_hinstance;
  newclass.lpszClassName  = "IUPSTATIC";
  test = RegisterClassEx(&newclass);
  assert(test != 0);
 
  memset(&newclass, 0, sizeof(WNDCLASSEX));
  newclass.cbSize         = sizeof(WNDCLASSEX);
  test = GetClassInfoEx(iupwin_hinstance, "LISTBOX", &newclass);
  assert(test != 0);
  newclass.hInstance      = iupwin_hinstance;
  newclass.lpszClassName  = "IUPLISTBOX";
  test = RegisterClassEx(&newclass);
  assert(test != 0);
 
  memset(&newclass, 0, sizeof(WNDCLASSEX));
  newclass.cbSize         = sizeof(WNDCLASSEX);
  test = GetClassInfoEx(iupwin_hinstance, "COMBOBOX", &newclass);
  assert(test != 0);
  newclass.hInstance      = iupwin_hinstance;
  newclass.lpszClassName  = "IUPCOMBOBOX";
  test = RegisterClassEx(&newclass);
  assert(test != 0);
}

static long gettextalignment(Ihandle *n)
{
  long alignment = ES_LEFT;
  char *type = IupGetAttribute(n, IUP_ALIGNMENT);

  if(!type)
    return alignment;

  if(strequal(type, IUP_ARIGHT))
    alignment = ES_RIGHT;
  else if(strequal(type, IUP_ALEFT))
    alignment = ES_LEFT;
  else if(strequal(type, IUP_ACENTER))
    alignment = ES_CENTER;

  return alignment;
}
